import { NextResponse } from 'next/server';
import { sendContactNotificationEmail } from '@/lib/email';


export async function POST(request) {
  try {
    const body = await request.json();
    const { name, email, phone, message} = body;

    if (!name || !email || !phone || !message) {
      return NextResponse.json(
        { 
          success: false,
          error: 'All fields are required' 
        },
        { status: 400 }
      );
    }
    const emailRegex = /^\w+([.-]?\w+)*@\w+([.-]?\w+)*(\.\w{2,3})+$/;
    if (!emailRegex.test(email)) {
      return NextResponse.json(
        { 
          success: false,
          error: 'Please enter a valid email address' 
        },
        { status: 400 }
      );
    }

    const phoneRegex = /^\+?[1-9]\d{1,14}$/;
    if (!phoneRegex.test(phone.replace(/\s/g, ''))) {
      return NextResponse.json(
        { 
          success: false,
          error: 'Please enter a valid phone number' 
        },
        { status: 400 }
      );
    }

    const contact = {
      name: name.trim(),
      email: email.toLowerCase().trim(),
      phone: phone.trim(),
      message: message.trim()
    };

    await sendContactNotificationEmail(contact);

    return NextResponse.json(
      { 
        success: true,
        message: 'Contact submission received successfully',
        data: contact 
      },
      { status: 201 }
    );
  } catch (error) {
    console.error('Create contact error:', error);
    
    if (error.name === 'ValidationError') {
      const errors = Object.values(error.errors).map(err => err.message);
      return NextResponse.json(
        { 
          success: false,
          error: 'Validation failed',
          details: errors 
        },
        { status: 400 }
      );
    }
    
    return NextResponse.json(
      { 
        success: false,
        error: 'Failed to submit contact form' 
      },
      { status: 500 }
    );
  }
}