import nodemailer from "nodemailer";

const sendContactNotificationEmail = async (contact) => {
  try {
    
    const transporter = nodemailer.createTransport({
      host: process.env.SENDER_HOST, // cPanel SMTP host
      port: 465,                   // 465 for SSL, 587 for TLS
      secure: true,                // true for 465, false for 587
      auth: {
        user: process.env.SENDER_EMAIL,
        pass: process.env.SENDER_EMAIL_PASSWORD      
      },
    });

    const contactDate = new Date().toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'long',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    });

    const mailOptions = {
      from: `"Books by Angela Brunson Website" <${process.env.SENDER_EMAIL}>`,
      to: process.env.ADMIN_EMAIL || process.env.SENDER_EMAIL,
      subject: `${contact.name} wants to connect with you`,
      text: `
        New Feedback - Books by Angela Brunson

        Customer: ${contact.name}
        Email: ${contact.email}
        Phone: ${contact.phone}

        Message:
        ${contact.message}

        Submitted: ${contactDate}

        Please respond to this inquiry within 24 hours.
              `.trim(),
              html: `
        <!DOCTYPE html>
        <html>
        <head>
          <meta charset="utf-8">
          <style>
            body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
            .container { max-width: 600px; margin: 0 auto; padding: 20px; }
            .header { background: #F7CB9F; color: white; padding: 20px; text-align: center; }
            .content { background: #353131ff; padding: 20px; }
            .info-box { background: white; padding: 15px; margin: 15px 0; border-radius: 5px; border-left: 4px solid #F7CB9F; }
            .urgent { background: #fff0f0; border-left: 4px solid #ff4444; }
            .action-box { background: #e8f4ff; padding: 15px; margin: 15px 0; border-radius: 5px; }
            .footer { text-align: center; padding: 20px; color: #666; font-size: 14px; }
            .btn { display: inline-block; padding: 10px 20px; background: #6699CC; color: white; text-decoration: none; border-radius: 5px; margin: 10px 0; }
          </style>
        </head>
        <body>
          <div class="container">
            <div class="header">
              <h1>Books by Angela Brunson</h1>
              <h2>New Feedback</h2>
            </div>

              <div class="info-box">
                <h3>Details</h3>
                <p><strong>Name:</strong> ${contact.name}</p>
                <p><strong>Email:</strong> <a href="mailto:${contact.email}">${contact.email}</a></p>
                <p><strong>Phone:</strong> <a href="tel:${contact.phone}">${contact.phone}</a></p>
                <p><strong>Submitted:</strong> ${contactDate}</p>
              </div>

              <div class="info-box">
                <h3>💬 Message</h3>
                <div style="background: #f8f9fa; padding: 15px; border-radius: 5px; border-left: 3px solid #6699CC;">
                  <p style="margin: 0; white-space: pre-wrap;">${contact.message}</p>
                </div>
              </div>
            
            <div class="footer">
              <p>&copy; ${new Date().getFullYear()} Books by Angela Brunson. All rights reserved.</p>
              <p>This is an automated notification from your website Books by Angela Brunson form.</p>
            </div>
          </div>
          </div>
        </body>
        </html>
      `,
      headers: {
        'Priority': 'high',
        'X-Priority': '1',
        'X-MSMail-Priority': 'High',
        'Importance': 'high'
      }
    };

    let retries = 3;
    while (retries > 0) {
      try {
        const info = await transporter.sendMail(mailOptions);
        console.log(`Contact notification email sent to admin: ${info.messageId}`);
        return true;
      } catch (sendError) {
        retries--;
        if (retries === 0) {
          throw sendError;
        }
        console.log(`Email send failed, retrying... (${retries} attempts left)`);
        await new Promise(resolve => setTimeout(resolve, 2000));
      }
    }

  } catch (error) {
    console.error("Error sending contact notification email:", error);
    throw new Error("Failed to send contact notification email");
  }
};

export { sendContactNotificationEmail };